import { useQuery } from "@tanstack/react-query";
import { Header } from "@/components/layout/header";
import { Footer } from "@/components/layout/footer";
import { HeroSection } from "@/components/ui/hero-section";
import { FeaturedBrands } from "@/components/ui/featured-brands";
import { ServiceCard } from "@/components/ui/service-card";
import { WhyChooseUs } from "@/components/ui/why-choose-us";
import { ProcessSection } from "@/components/ui/process-section";
import { TestimonialCard } from "@/components/ui/testimonial-card";
import { AboutSection } from "@/components/ui/about-section";
import { CTASection } from "@/components/ui/cta-section";
import { ContactSection } from "@/components/ui/contact-section";
import type { Service, Testimonial } from "@shared/schema";

export default function Home() {
  const { data: services = [] } = useQuery<Service[]>({
    queryKey: ['/api/services'],
  });

  const { data: testimonials = [] } = useQuery<Testimonial[]>({
    queryKey: ['/api/testimonials'],
  });

  const serviceCardColors = {
    jira: {
      borderColor: "bg-primary-600",
      color: "text-primary-600",
      iconBg: "bg-primary-50",
      iconColor: "text-primary-600",
      checkIconColor: "text-primary-600",
    },
    glpi: {
      borderColor: "bg-secondary-600",
      color: "text-secondary-600",
      iconBg: "bg-secondary-50",
      iconColor: "text-secondary-600",
      checkIconColor: "text-secondary-600",
    },
    esmp: {
      borderColor: "bg-accent-500",
      color: "text-accent-500",
      iconBg: "bg-accent-50",
      iconColor: "text-accent-500",
      checkIconColor: "text-accent-500",
    },
  };

  // Fallback service cards if API request fails
  const fallbackServices = [
    {
      id: 1,
      name: "Jira Service Management",
      slug: "jira",
      shortDescription: "Optimize your Jira Service Management instance with custom workflows, automation, and integrations.",
      iconClass: "fab fa-jira",
      colorClass: "text-primary-600",
      borderClass: "bg-primary-600",
      features: [
        "Custom workflow configuration",
        "Automation rule setup",
        "SLA & reporting customization",
        "Third-party app integration",
      ],
    },
    {
      id: 2,
      name: "GLPI Configuration",
      slug: "glpi",
      shortDescription: "Implement and customize GLPI to create a comprehensive IT asset and service management system.",
      iconClass: "fas fa-ticket-alt",
      colorClass: "text-secondary-600",
      borderClass: "bg-secondary-600",
      features: [
        "Asset management setup",
        "Helpdesk configuration",
        "Plugin installation & setup",
        "Custom dashboard creation",
      ],
    },
    {
      id: 3,
      name: "ESMP Implementation",
      slug: "esmp",
      shortDescription: "Configure Enterprise Service Management Platform to streamline cross-department service delivery.",
      iconClass: "fas fa-server",
      colorClass: "text-accent-500",
      borderClass: "bg-accent-500",
      features: [
        "Multi-department setup",
        "Process automation",
        "Self-service portal configuration",
        "Integration with existing systems",
      ],
    },
  ];

  // Fallback testimonials if API request fails
  const fallbackTestimonials = [
    {
      id: 1,
      name: "John Davis",
      position: "IT Director",
      company: "Global Finance Corp",
      testimonial: "The team's expertise in Jira Service Management completely transformed our IT helpdesk. The custom workflows they implemented have reduced our resolution times by over 40%.",
      rating: 5,
      initials: "JD",
      bgColor: "bg-primary-100",
      textColor: "text-primary-600",
    },
    {
      id: 2,
      name: "Sarah Mitchell",
      position: "CIO",
      company: "Healthcare Solutions",
      testimonial: "Their GLPI configuration services helped us implement a comprehensive asset management system that integrates perfectly with our service desk. The customization was exactly what we needed.",
      rating: 5,
      initials: "SM",
      bgColor: "bg-secondary-100",
      textColor: "text-secondary-600",
    },
    {
      id: 3,
      name: "Robert Kim",
      position: "Operations Manager",
      company: "Tech Innovations",
      testimonial: "The ESMP implementation was seamless, and the team's attention to detail was impressive. They understood our cross-departmental needs and configured the platform accordingly.",
      rating: 4,
      initials: "RK",
      bgColor: "bg-accent-100",
      textColor: "text-accent-600",
    },
  ];

  const displayServices = services.length > 0 ? services : fallbackServices;
  const displayTestimonials = testimonials.length > 0 ? testimonials : fallbackTestimonials;

  return (
    <>
      <Header />
      <main>
        <HeroSection />
        <FeaturedBrands />
        
        <section id="services" className="py-20 bg-slate-50 px-4 sm:px-6 lg:px-8">
          <div className="container mx-auto">
            <div className="text-center mb-16">
              <h2 className="text-3xl md:text-4xl font-bold mb-4">Our Configuration Services</h2>
              <p className="text-lg text-slate-600 max-w-2xl mx-auto">
                We specialize in tailoring service desk solutions to meet your organization's unique requirements.
              </p>
            </div>
            
            <div className="grid md:grid-cols-3 gap-8">
              {displayServices.map((service) => {
                const colorSet = serviceCardColors[service.slug as keyof typeof serviceCardColors];
                // Handle when features is not available from API
                const features = Array.isArray(service.features) 
                  ? service.features 
                  : fallbackServices.find(s => s.slug === service.slug)?.features || [];
                
                return (
                  <ServiceCard
                    key={service.id}
                    title={service.name}
                    description={service.shortDescription}
                    icon={service.iconClass}
                    features={features}
                    colorClass={service.colorClass}
                    borderColorClass={service.borderClass}
                    iconBgClass={colorSet.iconBg}
                    iconColorClass={colorSet.iconColor}
                    checkIconColorClass={colorSet.checkIconColor}
                    learnMoreLink={`/${service.slug}`}
                  />
                );
              })}
            </div>
          </div>
        </section>

        <WhyChooseUs />
        <ProcessSection />
        
        <section id="testimonials" className="py-20 bg-white px-4 sm:px-6 lg:px-8">
          <div className="container mx-auto">
            <div className="text-center mb-16">
              <h2 className="text-3xl md:text-4xl font-bold mb-4">What Our Clients Say</h2>
              <p className="text-lg text-slate-600 max-w-2xl mx-auto">
                Hear from organizations that have transformed their service management with our configuration services.
              </p>
            </div>
            
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
              {displayTestimonials.map((testimonial) => (
                <TestimonialCard
                  key={testimonial.id}
                  quote={testimonial.testimonial}
                  author={testimonial.name}
                  position={testimonial.position}
                  company={testimonial.company}
                  rating={testimonial.rating}
                  initials={testimonial.initials}
                  bgColor={testimonial.bgColor}
                  textColor={testimonial.textColor}
                />
              ))}
            </div>
          </div>
        </section>
        
        <AboutSection />
        <CTASection />
        <ContactSection />
      </main>
      <Footer />
    </>
  );
}
